import xbmc
from urllib.request import Request, urlopen
from urllib.parse import quote
from urllib.error import URLError
import time
import json
import telnetlib

# Volume map: Maps /Volumes/X/ to the server's expected path
VOLUME_MAP = {
    "3": "/mnt/sda1/Movies3/",
    "4": "/mnt/sda1/Movies4/",
    "5": "/mnt/sda1/Movies5/",
    "7": "/mnt/sda1/Movies7/",
    "8": "/mnt/sda1/Movies8/"
}

# Sony Bravia TV configuration
BRAVIA_IP = "192.168.1.55"
BRAVIA_PSK = "0000"

# Oppo device HTTP configuration
OPPO_IP = "192.168.1.57"
OPPO_PORT = 436

def wake_oppo():
    """Wake the Oppo via Telnet using root:oppo credentials."""
    try:
        tn = telnetlib.Telnet(OPPO_IP, 23, timeout=5)
        tn.read_until(b"login: ", 5)
        tn.write(b"root\n")
        tn.read_until(b"Password: ", 5)
        tn.write(b"oppo\n")
        tn.read_until(b"#", 5)
        tn.write(b"UPW 1\n")  # Power on command (per April 16, 2025)
        response = tn.read_until(b"OK", 5).decode('utf-8', errors='ignore')
        tn.close()
        xbmc.log(f"Oppo Telnet response: {response}", xbmc.LOGINFO)
        if "OK" in response:
            xbmc.log("Oppo powered on via Telnet", xbmc.LOGINFO)
            time.sleep(3)  # Increased delay for Oppo initialization
            return True
        else:
            xbmc.log("Oppo Telnet command failed, response did not contain OK", xbmc.LOGERROR)
            return False
    except Exception as e:
        xbmc.log(f"Failed to wake Oppo via Telnet: {e}", xbmc.LOGERROR)
        return False

# ... (keep switch_to_hdmi4, switch_to_hdmi1, check_oppo_playback_status, monitor_oppo_playback unchanged) ...

class MyPlayer(xbmc.Player):
    def __init__(self):
        xbmc.Player.__init__(self)
        xbmc.log("MyPlayer initialized", xbmc.LOGINFO)

    def onPlayBackStarted(self):
        xbmc.log("Playback started detected", xbmc.LOGINFO)
        xbmc.sleep(500)

        try:
            file_path = self.getPlayingFile()
            xbmc.log(f"Original file path: {file_path}", xbmc.LOGINFO)

            normalized_path = file_path
            if file_path.startswith("nfs://192.168.1.6/Volumes/"):
                parts = file_path.split("/")
                if len(parts) > 4:
                    volume_num = parts[4]  # Extract volume number from nfs://192.168.1.6/Volumes/X/
                    if volume_num in VOLUME_MAP:
                        # Replace NFS prefix with mapped path
                        normalized_path = VOLUME_MAP[volume_num] + "/".join(parts[5:])
                        xbmc.log(f"Normalized file path: {normalized_path}", xbmc.LOGINFO)
                    else:
                        xbmc.log(f"Volume {volume_num} not found in VOLUME_MAP, skipping request", xbmc.LOGERROR)
                        return
                else:
                    xbmc.log("Invalid NFS path structure, skipping request", xbmc.LOGERROR)
                    return
            else:
                xbmc.log("File path not under expected NFS /Volumes/, cannot normalize", xbmc.LOGERROR)
                return

            # Verify Oppo HTTP endpoint is reachable
            try:
                req = Request(f"http://{OPPO_IP}:{OPPO_PORT}/getmovieplayinfo")
                with urlopen(req, timeout=5) as response:
                    xbmc.log(f"Oppo HTTP endpoint check: status code {response.getcode()}", xbmc.LOGINFO)
            except URLError as e:
                xbmc.log(f"Oppo HTTP endpoint unreachable: {e}", xbmc.LOGERROR)
                return

            # Wake Oppo before sending request
            if not wake_oppo():
                xbmc.log("Skipping playback request due to Oppo wake failure", xbmc.LOGERROR)
                return

            switch_to_hdmi4()

            json_payload = f'{{"path":"{normalized_path}","index":0,"type":1,"appDeviceType":2,"extraNetPath":"192.168.1.238/Storage/UHD/","playMode":0}}'
            encoded_payload = quote(json_payload)
            url = f"http://{OPPO_IP}:{OPPO_PORT}/playnormalfile?{encoded_payload}"
            xbmc.log(f"Constructed URL: {url}", xbmc.LOGINFO)

            for attempt in range(3):  # Increased to 3 attempts
                try:
                    req = Request(url)
                    xbmc.log(f"Sending HTTP request (attempt {attempt + 1})", xbmc.LOGINFO)
                    with urlopen(req, timeout=5) as response:
                        status_code = response.getcode()
                        response_content = response.read().decode('utf-8', errors='ignore')
                        xbmc.log(f"HTTP response: status code: {status_code}, content: {response_content}", xbmc.LOGINFO)
                        
                        if '"success":true' in response_content:
                            xbmc.log("Server confirmed successful playback", xbmc.LOGINFO)
                            # Stop Kodi playback
                            self.stop()
                            xbmc.executebuiltin("PlayerControl(Stop)")
                            for _ in range(10):  # Increased retries
                                if not self.isPlaying():
                                    xbmc.log("Kodi playback stopped", xbmc.LOGINFO)
                                    break
                                xbmc.sleep(500)
                            else:
                                xbmc.log("Failed to stop Kodi playback", xbmc.LOGERROR)
                            monitor_oppo_playback()
                            return
                        else:
                            xbmc.log(f"Server rejected request: {response_content}", xbmc.LOGWARNING)
                            if attempt < 2:
                                xbmc.log("Retrying after 2-second delay", xbmc.LOGINFO)
                                xbmc.sleep(2000)
                            else:
                                xbmc.log("Final retry failed", xbmc.LOGERROR)
                except URLError as e:
                    xbmc.log(f"HTTP request failed: {e}", xbmc.LOGERROR)
                    if attempt < 2:
                        xbmc.log("Retrying after 1-second delay", xbmc.LOGINFO)
                        xbmc.sleep(1000)
                    else:
                        xbmc.log("Final retry failed due to network error", xbmc.LOGERROR)
            # If playback fails, stop Kodi and switch back
            self.stop()
            xbmc.executebuiltin("PlayerControl(Stop)")
            switch_to_hdmi1()
        except Exception as e:
            xbmc.log(f"Error getting or processing file: {e}", xbmc.LOGERROR)
            self.stop()
            xbmc.executebuiltin("PlayerControl(Stop)")
            switch_to_hdmi1()

# ... (keep MyMonitor and main loop unchanged) ...